<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Admin\ACommonController;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportManagementController extends ACommonController
{

    public function DriverReport(Request $request)
    {
        $query = DB::table('booking_management')
            ->join('driver_management', 'booking_management.driver_id', '=', 'driver_management.id')
            ->where('booking_management.trip_status', 0)
            ->select(
                'booking_management.id as booking_id',
                'booking_management.driver_id',
                'driver_management.driver_name',
                'booking_management.booking_date'
            );

        if ($request->from_date && $request->to_date) {
            $query->whereBetween('booking_date', [$request->from_date, $request->to_date]);
        }

        $bookings = $query->get();

        // ✅ Group by driver with trip list
        $driverReports = $bookings->groupBy('driver_id')->map(function ($trips, $driverId) {
            $bookingIds = $trips->pluck('booking_id');

            $totalAmount = DB::table('fare_summaries')
                ->whereIn('booking_id', $bookingIds)
                ->sum('total_price');

            $tripDetails = $trips->map(function ($trip) {
                return [
                    'booking_id'   => $trip->booking_id,
                    'booking_date' => $trip->booking_date,
                ];
            });

            return [
                'driver_id'    => $driverId,
                'driver_name'  => $trips->first()->driver_name,
                'trip_count'   => $trips->count(),
                'total_amount' => $totalAmount,
                'trips'        => $tripDetails,
            ];
        });

        // ✅ If PDF requested
        if ($request->has('download') && $request->download === 'pdf') {
            $pdf = Pdf::loadView('admin.report_management.driver_report_pdf', [
                'driverReports' => $driverReports,
                'from_date'     => $request->from_date,
                'to_date'       => $request->to_date,
            ]);
            return $pdf->download('driver_report.pdf');
        }

        // ✅ Default HTML view
        return view('admin.report_management.driver_report', compact('driverReports'));
    }

    public function CustomerReport(Request $request)
    {
        $query = DB::table('booking_management')
            ->where('trip_status', 0)
            ->whereNotNull('phone')
            ->select(
                'id as booking_id',
                'customer_name',
                'phone',
                'booking_date'
            );

        // 📅 Date filter
        if ($request->from_date && $request->to_date) {
            $query->whereBetween('booking_date', [$request->from_date, $request->to_date]);
        }

        // 🔍 Name filter
        if ($request->customer_name) {
            $query->where('customer_name', 'like', '%' . $request->customer_name . '%');
        }

        $bookings = $query->get();

        // ✅ Group by phone, include trip list and total fare
        $customerReports = $bookings->groupBy('phone')->map(function ($trips, $phone) {
            $bookingIds = $trips->pluck('booking_id');

            $totalAmount = DB::table('fare_summaries')
                ->whereIn('booking_id', $bookingIds)
                ->sum('total_price');

            // Individual trip list with booking date
            $tripDetails = $trips->map(function ($trip) {
                return [
                    'booking_id'   => $trip->booking_id,
                    'booking_date' => $trip->booking_date,
                ];
            });

            return [
                'customer_name' => $trips->first()->customer_name,
                'phone'         => $phone,
                'trip_count'    => $trips->count(),
                'total_amount'  => $totalAmount,
                'trips'         => $tripDetails,
            ];
        });

        // 📄 PDF export
        if ($request->has('download') && $request->download === 'pdf') {
            $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('admin.report_management.customer_report_pdf', [
                'customerReports' => $customerReports,
                'from_date'       => $request->from_date,
                'to_date'         => $request->to_date,
                'customer_name'   => $request->customer_name,
            ]);
            return $pdf->download('customer_report.pdf');
        }

        return view('admin.report_management.customer_report', compact('customerReports'));
    }

}
