<?php
namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use App\Models\Profile;
use App\Models\OrderAddon;
use App\Models\Addon;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\OrderMeasurement;
use App\Models\Customer;
use Illuminate\Http\Request;

class OrderManagementController extends Controller
{

public function index(Request $request)
{
    $search = $request->query('search');

    $orders = Order::with(['customer', 'items'])
        ->when($search, function ($query, $search) {
            $query->where('order_no', 'like', "%{$search}%")
                  ->orWhereHas('customer', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%");
                  });
        })
        ->orderBy('id', 'desc')
        ->paginate(10)
        ->withQueryString(); // 🔑 keep search while paging

    return view('admin.order_management.list', compact(
        'orders',
        'search'
    ));
}


    public function create( )
    {
         $customers = Customer::all();
         $addons    = Addon::all();
        return view('admin.order_management.add', compact('customers','addons'));
    }



public function store(Request $request)
{
    DB::beginTransaction();

    try {


        /* ================= VALIDATION ================= */
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'delivery_date' => 'required|date',
            'garments' => 'required|array',
            'garments.*.garment_type' => 'required|string',
            'garments.*.profile_id' => 'required|exists:profiles,id',
            'garments.*.unit_price' => 'required|numeric|min:0',
            'garments.*.quantity' => 'nullable|integer|min:1',
            'garments.*.discount' => 'nullable|numeric|min:0',
        ]);
     /* ================= GET GARMENTS AS ARRAY ================= */
$garments = $request->input('garments', []);

/* ================= CREATE / GET SELF PROFILE ================= */
$selfProfile = Profile::firstOrCreate(
    [
        'customer_id' => $request->customer_id,
        'relation'    => 'Self',
    ],
    [
        'profile_name' => 'Self',
    ]
);

/* ================= MAP "self" → REAL PROFILE ID ================= */
foreach ($garments as $index => $g) {
    if ($g['profile_id'] === 'self') {
        $garments[$index]['profile_id'] = $selfProfile->id;
    }
}

/* ================= MERGE BACK INTO REQUEST ================= */
$request->merge([
    'garments' => $garments
]);
        /* ================= ORDER ================= */
        $order = Order::create([
            'customer_id'    => $request->customer_id,
            'advance_amount' => $request->advance_amount ?? 0,
            'delivery_date'  => $request->delivery_date,
        ]);

        // Generate readable order number
        $order->update([
            'order_no' => now()->year . '-' . $order->id
        ]);

        /* ================= GARMENTS LOOP ================= */
        foreach ($request->garments as $g) {

            // 🔒 Validate profile belongs to customer
            $profile = Profile::where('id', $g['profile_id'])
                ->where('customer_id', $order->customer_id)
                ->firstOrFail();

            /* ================= IMAGES ================= */
            $imagePaths = [];

            if (!empty($g['images']) && is_array($g['images'])) {
                foreach ($g['images'] as $image) {
                    $imagePaths[] = $image->store('order_images', 'public');
                }
            }

            /* ================= NORMALIZE VALUES ================= */
            $unitPrice = (float) $g['unit_price'];
            $qty       = isset($g['quantity']) ? (int) $g['quantity'] : 1;
            $discount  = isset($g['discount']) ? (float) $g['discount'] : 0;

            /* ================= ORDER ITEM ================= */
            $orderItem = OrderItem::create([
                'order_id'        => $order->id,
                'profile_id'      => $profile->id,
                'garment_type'    => $g['garment_type'],
                'reference_image' => !empty($imagePaths) ? json_encode($imagePaths) : null,
                'notes'           => $g['notes'] ?? null,
                'unit_price'      => $unitPrice,
                'quantity'        => $qty,
                'discount'        => $discount,
                'total_amount'    => 0, // calculated later
            ]);

            /* ================= MEASUREMENTS ================= */
            if (!empty($g['measurement']) && is_array($g['measurement'])) {
                OrderMeasurement::create([
                    'order_item_id' => $orderItem->id,
                    'measurements'  => $g['measurement'],
                ]);
            }

            /* ================= ADD-ONS ================= */
            $addonTotal    = 0;
            $addonSnapshot = [];

            if (!empty($g['addons']) && is_array($g['addons'])) {

                foreach ($g['addons'] as $addon) {

                    $masterAddon = Addon::findOrFail($addon['addon_id']);

                    $addonSnapshot[] = [
                        'addon_id'    => $masterAddon->id,
                        'addon_name'  => $masterAddon->name,
                        'addon_price' => $masterAddon->price,
                    ];

                    $addonTotal += $masterAddon->price;
                }

                OrderAddon::create([
                    'order_id'      => $order->id,
                    'profile_id'    => $profile->id,
                    'order_item_id' => $orderItem->id,
                    'addon_values'  => json_encode($addonSnapshot),
                    'addon_amount'  => $addonTotal,
                ]);
            }

            /* ================= FINAL TOTAL ================= */
        $finalTotal = ($unitPrice * $qty) + $addonTotal - $discount;


            $orderItem->update([
                'total_amount' => max($finalTotal, 0)
            ]);
        }

        DB::commit();

        return response()->json([
            'success'  => true,
            'redirect' => route('admin.OrderManagement-index')
        ]);

    } catch (\Throwable $e) {

        DB::rollBack();

        return response()->json([
            'success' => false,
            'message' => $e->getMessage()
        ], 500);
    }
}

    public function show($id)
    {
        $order = Order::with('items.measurement', 'customer')
            ->findOrFail($id);

        return view('admin.order_management.view', compact('order'));
    }


    public function edit($id)
    {
        $order = Order::with('items.measurement')
            ->findOrFail($id);

        $customers = Customer::all();

        return view('admin.order_management.edit', compact('order', 'customers'));
    }

 public function update(Request $request, $id)
{
    DB::beginTransaction();

    try {

        /* ================= VALIDATION ================= */
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'delivery_date' => 'required|date',
            'garments' => 'required|array',
            'garments.*.garment_type' => 'required|string',
            'garments.*.profile_id' => 'required',
            'garments.*.unit_price' => 'required|numeric|min:0',
            'garments.*.quantity' => 'nullable|integer|min:1',
            'garments.*.discount' => 'nullable|numeric|min:0',
        ]);

        /* ================= ORDER ================= */
        $order = Order::findOrFail($id);

        $order->update([
            'customer_id'    => $request->customer_id,
            'advance_amount' => $request->advance_amount ?? 0,
            'delivery_date'  => $request->delivery_date,
        ]);

        /* ================= CLEAN OLD DATA ================= */
        foreach ($order->items as $item) {
            OrderAddon::where('order_item_id', $item->id)->delete();
            OrderMeasurement::where('order_item_id', $item->id)->delete();
        }

        OrderItem::where('order_id', $order->id)->delete();

        /* ================= RE-CREATE ITEMS ================= */
        foreach ($request->garments as $g) {

            // 🔒 Validate profile belongs to customer
            $profile = Profile::where('id', $g['profile_id'])
                ->where('customer_id', $order->customer_id)
                ->firstOrFail();

            /* ================= IMAGES ================= */
            $imagePaths = [];
            if (!empty($g['images']) && is_array($g['images'])) {
                foreach ($g['images'] as $image) {
                    $imagePaths[] = $image->store('order_images', 'public');
                }
            }

            /* ================= NORMALIZE VALUES ================= */
            $unitPrice = (float) $g['unit_price'];
            $qty       = isset($g['quantity']) ? (int) $g['quantity'] : 1;
            $discount  = isset($g['discount']) ? (float) $g['discount'] : 0;

            /* ================= ORDER ITEM ================= */
            $orderItem = OrderItem::create([
                'order_id'        => $order->id,
                'profile_id'      => $profile->id,
                'garment_type'    => $g['garment_type'],
                'reference_image' => !empty($imagePaths) ? json_encode($imagePaths) : null,
                'notes'           => $g['notes'] ?? null,
                'unit_price'      => $unitPrice,
                'quantity'        => $qty,
                'discount'        => $discount,
                'total_amount'    => 0, // calculated below
            ]);

            /* ================= MEASUREMENTS ================= */
            if (!empty($g['measurement']) && is_array($g['measurement'])) {
                OrderMeasurement::create([
                    'order_item_id' => $orderItem->id,
                    'measurements'  => $g['measurement'],
                ]);
            }

            /* ================= ADD-ONS ================= */
            $addonTotal    = 0;
            $addonSnapshot = [];

            if (!empty($g['addons']) && is_array($g['addons'])) {

                foreach ($g['addons'] as $addon) {

                    $masterAddon = Addon::findOrFail($addon['addon_id']);

                    $addonSnapshot[] = [
                        'addon_id'    => $masterAddon->id,
                        'addon_name'  => $masterAddon->name,
                        'addon_price' => $masterAddon->price,
                    ];

                    $addonTotal += $masterAddon->price;
                }

                OrderAddon::create([
                    'order_id'      => $order->id,
                    'profile_id'    => $profile->id,
                    'order_item_id' => $orderItem->id,
                    'addon_values'  => json_encode($addonSnapshot),
                    'addon_amount'  => $addonTotal,
                ]);
            }

            /* ================= FINAL TOTAL ================= */
          $finalTotal = ($unitPrice * $qty) + $addonTotal - $discount;


            $orderItem->update([
                'total_amount' => max($finalTotal, 0)
            ]);
        }

        DB::commit();

        return redirect()
            ->route('admin.OrderManagement-index')
            ->with('success', 'Order updated successfully');

    } catch (\Throwable $e) {

        DB::rollBack();

        return back()->withErrors($e->getMessage());
    }
}



    public function saveSingleCustomer(Request $request)
    {
    $request->validate([
        'customer_id'   => 'nullable|exists:customers,id',
        'customer_name' => 'required|string',
        'profile_name'  => 'required|string',
        'garment_type'  => 'required|string',
        'measurement'   => 'array'
    ]);

    // Example save (adjust tables as needed)
    $orderItem = [
        'customer_id'   => $request->customer_id,
        'customer_name' => $request->customer_name,
        'profile_name'  => $request->profile_name,
        'garment_type'  => $request->garment_type,
        'measurement'   => $request->measurement
    ];


    return response()->json([
        'success' => true,
        'data'    => $orderItem
    ]);
    }

    public function destroy($id)
    {
        Order::findOrFail($id)->delete();

        return redirect()->route('admin.OrderManagement-index')
            ->with('success', 'Order Deleted Successfully');
    }
public function print($id)
{
    $order = Order::with([
        'customer',
        'items.profile',
        'items.measurement',
        'items.addons'
    ])->findOrFail($id);

    return view('admin.order_management.print', compact('order'));
}
public function cloneList($customerId, $garment)
{
    $orders = Order::with('items.measurement')
        ->where('customer_id', $customerId)
        ->whereHas('items', function ($q) use ($garment) {
            $q->where('garment_type', $garment);
        })
        ->latest()
        ->take(10)
        ->get()
        ->map(function ($order) {
            $order->items = $order->items->map(function ($item) {
                return [
                    'garment_type' => $item->garment_type,
                    'measurement'  => $item->measurement?->measurements ?? [],
                ];
            });
            return $order;
        });

    return response()->json($orders);
}
public function getPrice(Request $request)
{
    $request->validate([
        'garment' => 'required|string',
        'age'     => 'required|string',
    ]);

    $price = DB::table('settings')
        ->where('garment_type', $request->garment)
        ->where('age_type', $request->age)
        ->value('price');
    // dd($price);
    return response()->json([
        'price' => $price ?? 0
    ]);
}

public function getLatestMeasurement($profileId, $garment)
{
    $item = OrderItem::where('profile_id', $profileId)
        ->where('garment_type', $garment)
        ->whereHas('measurement')
        ->latest()
        ->first();

    if (! $item || ! $item->measurement) {
        return response()->json(['measurements' => null]);
    }

    return response()->json([
        'measurements' => $item->measurement->measurements
    ]);
}


public function getCustomerProfiles($customerId)
{
    $customer = Customer::findOrFail($customerId);


    $selfProfile = Profile::updateOrCreate(
        [
            'customer_id' => $customer->id,
            'relation'    => 'Self',
        ],
        [
            'profile_name' => $customer->name, // 🔥 SYNC NAME
        ]
    );

    $profiles = Profile::where('customer_id', $customerId)
        ->select('id', 'profile_name', 'relation')
        ->get();

    return response()->json([
        'data' => $profiles,
        'self_profile_id' => $selfProfile->id
    ]);
}


public function latestMeasurements(Request $request)
{
    $request->validate([
        'customer'   => 'required|exists:customers,id',
        'profile_id' => 'required|exists:profiles,id',
        'garment'    => 'required|string',
    ]);

    $measurement = OrderMeasurement::whereIn(
        'order_item_id',
        OrderItem::where('garment_type', $request->garment)
            ->where('profile_id', $request->profile_id) // ✅ IMPORTANT
            ->whereHas('order', function ($q) use ($request) {
                $q->where('customer_id', $request->customer);
            })
            ->pluck('id')
    )
    ->latest()
    ->first();


    if (! $measurement) {
        return response()->json(['data' => null]);
    }

    return response()->json([
        'data' => $measurement->measurements
    ]);
}

public function storeProfile(Request $request)
{
    $request->validate([
        'customer_id' => 'required|exists:customers,id',
        'profile_name'        => 'required|string|max:100',
        'relation'    => 'nullable|string|max:50',
    ]);

    $profile = Profile::create([
        'customer_id' => $request->customer_id,
        'profile_name'        => $request->profile_name,
        'relation'    => $request->relation,
    ]);

    return response()->json([
        'success' => true,
        'data'    => $profile
    ]);


}
public function customerReceipt(Order $order)
{
    // 🔹 Load required relations (important)
    $order->load([
        'customer',
        'items.profile',
        'items.measurement'
    ]);

    // 🔹 Return customer receipt blade
    return view('admin.order_management.customer-receipt', compact('order'));
}

 public function whatsappLink(Order $order)
{
    $phone = preg_replace('/\D/', '', $order->customer->phone);

    if (strlen($phone) === 10) {
        $phone = '91' . $phone;
    }

    $fileUrl = route('OrderManagement-customerReceipt', $order->id);

    $message =
        "Dear {$order->customer->name},\n\n" .
        "Order No: {$order->order_no}\n" .
        "Order Date: {$order->created_at->format('d-m-Y')}\n" .
        "Delivery Date: " . \Carbon\Carbon::parse($order->delivery_date)->format('d-m-Y') . "\n" .
        "Total Amount: ₹" . number_format($order->items->sum('total_amount'), 2) . "\n\n" .
        "View order & cloth images:\n" .
        $fileUrl . "\n\n" .
        "Thank you.";

    return redirect(
        "https://wa.me/{$phone}?text=" . urlencode($message)
    );
}
public function generateReceiptPdf($orderId)
{
    $order = Order::with('customer','items')->findOrFail($orderId);

    $filePath = public_path(
        'storage/receipts/customer_order_'.$order->id.'.pdf'
    );

    // Generate only if not exists (optional but good)
    if (!file_exists($filePath)) {

        $pdf = Pdf::loadView(
            'admin.order_management.receipt_pdf',
            compact('order')
        );

        Storage::disk('public')->put(
            'receipts/customer_order_'.$order->id.'.pdf',
            $pdf->output()
        );
    }

    // ✅ FORCE OPEN PDF
    return response()->file($filePath);
}



}
